/* FilteredScrollableGraphTableComponent.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.gui.components;

import java.awt.Color;
import java.awt.Component;
import javax.swing.BorderFactory;
import javax.swing.JList;
import javax.swing.JViewport;
import javax.swing.ListSelectionModel;
import javax.swing.UIManager;
import javax.swing.border.Border;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import org.grinvin.conjecture.filter.GraphFilter;
import org.grinvin.conjecture.filter.GraphFilterListener;
import org.grinvin.invariants.InvariantValue;
import org.grinvin.gui.components.render.FilteredInvariantValueTableCellRenderer;
import org.grinvin.gui.components.render.GraphListElementRenderer;
import org.grinvin.list.graphs.GraphList;
import org.grinvin.list.graphs.GraphListModel;
import org.grinvin.list.invariants.InvariantListModel;

/**
 * A version of {@link ScrollableGraphTableComponent} that displays a filter as well.
 */
public class FilteredScrollableGraphTableComponent extends ScrollableGraphTableComponent implements GraphFilterListener {
    
    //
    private GraphFilter graphFilter;
    
    //
    private GraphList graphList;
    
    //
    public FilteredScrollableGraphTableComponent(GraphListModel graphListModel, InvariantListModel invariantListModel, ListSelectionModel selectionModel, GraphFilter graphFilter) {
        super(graphListModel, invariantListModel, selectionModel);
        this.graphFilter = graphFilter;
        this.graphList = graphListModel;
        
        graphFilter.addGraphFilterListener(this);
        
        // row header view is an almost standard graph list component
        GraphListComponent rowHeaderView =
                new GraphListComponent(graphListModel, selectionModel);
        rowHeaderView.setFixedCellHeight(graphtable.getRowHeight());
        rowHeaderView.setCellRenderer(new FilteredTabelHeaderCellRenderer()); // overrides cell renderer for graph list component
        rowHeaderView.setPaintWatermark(false);
        setRowHeaderView(rowHeaderView);
        //workaround from http://bugs.sun.com/bugdatabase/view_bug.do?bug_id=4202002
        getRowHeader().addChangeListener(new ChangeListener() {
              public void stateChanged(ChangeEvent e) {
                JViewport jvViewport = (JViewport) e.getSource();
                int iExtent = jvViewport.getExtentSize().height;
                int iMax = jvViewport.getViewSize().height;
                int iValue = Math.max(0, Math.min(jvViewport.getViewPosition().y, iMax - iExtent));
                getVerticalScrollBar().setValues(iValue, iExtent, 0, iMax);
              }
            });

        graphtable.setDefaultRenderer(InvariantValue.class, new FilteredInvariantValueTableCellRenderer(graphFilter, graphListModel));

    }

    public void filterChanged(GraphFilter filter) {
        repaint();
    }
    
    //
    private class FilteredTabelHeaderCellRenderer extends GraphListElementRenderer {
        //
        private final Border noFocusBorder;
        
        //
        private final Color red = new Color(255, 0, 0, 128);

        private final Color darkerRed;
                        
        public FilteredTabelHeaderCellRenderer() {
            this.noFocusBorder = UIManager.getBorder("TableHeader.cellBorder");
            darkerRed = new Color(red.darker().getRed(), 0, 0, 128);
        }
        
        public Component getListCellRendererComponent
                (JList list, Object value, int index, boolean isSelected, boolean cellHasFocus) {
            
            super.getListCellRendererComponent(list, value, index, isSelected, cellHasFocus);
            boolean accepted = graphFilter.accepts(graphList.get(index).getBundle());
            if (! cellHasFocus && ! isSelected && accepted) //fixes incorrect background on OS X
                setBorder(BorderFactory.createCompoundBorder(noFocusBorder, BorderFactory.createEmptyBorder(0,0,0,5)));
            if(! accepted){
                setOpaque(true);
                setBackground(isSelected ? darkerRed : red);
            }
            return this;
        }
        
    }
}
