/* ChangeCoordinates.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.gred.undoable;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;

import org.grinvin.graphs.Embedding;
import org.grinvin.graphs.Vertex;

/**
 * Undoable change for moving elements in a graph.
 */
public class ChangeCoordinates extends InternationalizedUndoableChange {
    
    /**
     * Embedding in which the move was performed.
     */
    private Embedding embedding;
    
    /**
     * Represents a change of coordinates for a single vertex.
     */
    private static class Movement {
        
        //
        private Vertex vertex;
        
        //
        private double[] oldCoordinates;
        
        //
        private double[] newCoordinates;
        
    }
    
    /**
     * Collection of movements made.
     */
    private Collection<Movement> movements;
    
    /**
     * Create an undoable change object for moving elements in an embedding.
     * Individual movements should be added later with {@link #addMovement}.
     * @param embedding Corresponding embedding
     * @param key Resource key for the undo/redo captions for this move
     */
    public ChangeCoordinates (Embedding embedding, String key) {
        super (key);
        this.embedding = embedding;
        this.movements = new ArrayList<Movement> ();
    }
    
    /**
     * Add a movement of the given vertex form old to new coordinates. If boths sets
     * of coordinates are exactly the same, then no operation is performed.
     */
    public void addMovement (Vertex v, double[] oldCoordinates, double[] newCoordinates) {
        if (! Arrays.equals (oldCoordinates, newCoordinates)) {
            Movement m = new Movement ();
            m.vertex = v;
            m.oldCoordinates = oldCoordinates;
            m.newCoordinates = newCoordinates;
            movements.add (m);
        }
    }
    
    
    /**
     * Undoes the effect of this change by moving all vertices back to their
     * original position.
     */
    public void undo () {
        for (Movement m: movements) {
            embedding.setCoordinates (m.vertex, m.oldCoordinates);
        }
    }
    
    /**
     * Revertes an undo by moving all vertices to their new positions.
     */
    public void redo () {
        for (Movement m: movements) {
            embedding.setCoordinates (m.vertex, m.newCoordinates);
        }
    }
    
    
    
}
