/* DefaultEmbeddingModel.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.graphs;

import java.util.Arrays;

/**
 * Implementation of {@link Embedding} that at the same time functions as a model
 * in a model-view-controller triple. An {@link EmbeddingListener} may
 * register itself with this model to be informed about various changes:
 * <ul>
 * <li>Vertex coordinate changes.</li>
 * <li>Dimension changes of the embedding.</li>
 * </ul><p>
 * Note that changes to the peer of this embedding (vertices being added or removed)
 * should be listened to by a separate {@link EmbeddingListener} registered with the peer.
 *
 */
public class DefaultEmbeddingModel extends DefaultEmbedding implements MutableEmbeddingModel {
    
    //
    private EmbeddingModelSupport support;
    
    /**
     * Create an embedding of the given dimension for the given peer graph.
     */
    public DefaultEmbeddingModel(GraphView peer, int dimension) {
        super(peer, dimension);
        this.support = new EmbeddingModelSupport();
    }
    
    // extends DefaultEmbedding
    @Override public void setCoordinates(Vertex vertex, double[] coordinates) {
        double[] oldCoordinates = getCoordinates(vertex);
        if (! Arrays.equals(coordinates, oldCoordinates)) {
            super.setCoordinates(vertex, coordinates);
            support.fireCoordinatesChanged(vertex);
        }
    }
    
    // extends DefaultEmbedding
    @Override public void setDimension(int dimension) {
        if (this.dimension != dimension) {
            int oldDimension = this.dimension;
            super.setDimension(dimension);
            support.fireDimensionChanged(oldDimension);
        }
    }
    
    // implements MutableEmbeddingModel
    public void setCoordinates(Vertex[] vertices, double[][] coordinates) {
        for (int i=0; i < vertices.length; i++)
            super.setCoordinates(vertices[i],  coordinates[i]);
        support.fireEmbeddingChanged();
    }
    
    // implements EmbeddingModel
    public void removeEmbeddingListener(EmbeddingListener l) {
        support.removeEmbeddingListener(l);
    }
    
    // implements EmbeddingModel
    public void addEmbeddingListener(EmbeddingListener l) {
        support.addEmbeddingListener(l);
    }
    
    // extends DefaultEmbedding
    @Override public void copy(EmbeddingView original, GraphView newPeer) {
        super.copy(original, newPeer);
        support.fireEmbeddingChanged();
    }
    
    // extends DefaultEmbedding
    @Override public void copy(EmbeddingView original) {
        super.copy(original);
        support.fireEmbeddingChanged();
    }
    
    
    
}
